<?php

namespace App\Http\Controllers;

use App\Models\Role;
use App\Models\Company;
use Illuminate\Http\Request;

class RoleController extends Controller
{
    public function index(Request $request)
    {
        $user = auth()->user();
        $tenant = $user?->company_code ?? session('company_code');
        $role = $user?->role;

        // Check read permission (Superadmin 000 bypasses check)
        if ($tenant !== '000') {
            if (!$role || !$role->hak_akses_tbl_role_read) {
                abort(403, 'Anda tidak memiliki akses untuk melihat data role.');
            }
        }

        $limit = $request->input('limit', 40);
        $search = $request->input('search');

        $roles = Role::with('company')
            ->when($search, function ($query, $search) {
                return $query->where('nama_role', 'like', "%{$search}%")
                    ->orWhere('company_code', 'like', "%{$search}%")
                    ->orWhere('keterangan', 'like', "%{$search}%")
                    ->orWhereHas('company', function ($q) use ($search) {
                        $q->where('nama_company', 'like', "%{$search}%");
                    });
            })
            ->when($tenant && $tenant !== '000', function ($q) use ($tenant) {
                $q->where('company_code', $tenant);
            })
            ->paginate($limit)
            ->withQueryString();

        // Permission Checks
        if ($tenant === '000') {
            $canCreate = true;
            $canEdit = true;
            $canDelete = true;
        } else {
            $canCreate = ($role && $role->hak_akses_tbl_role_cread);
            $canEdit = ($role && $role->hak_akses_tbl_role_edit);
            $canDelete = ($role && $role->hak_akses_tbl_role_delete);
        }

        $breadcrumb = [
            ['name' => 'Home', 'url' => route('dashboard.index')],
            ['name' => 'Role', 'url' => ''],
        ];

        return view('role.index', compact('roles', 'breadcrumb', 'canCreate', 'canEdit', 'canDelete'));
    }

    public function create()
    {
        $user = auth()->user();
        $tenant = $user->company_code ?? session('company_code');
        $role = $user->role;

        // Check create permission
        if ($tenant !== '000') {
            if (!$role || !$role->hak_akses_tbl_role_cread) {
                return redirect()->route('role.index')->with('error', 'Anda tidak memiliki akses untuk menambah role.');
            }
        }

        $companies = Company::when($tenant && $tenant !== '000', function ($q) use ($tenant) {
            $q->where('company_code', $tenant);
        })->get();
        $breadcrumb = [
            ['name' => 'Home', 'url' => route('dashboard.index')],
            ['name' => 'Role', 'url' => route('role.index')],
            ['name' => 'Tambah', 'url' => ''],
        ];

        return view('role.create', compact('companies', 'breadcrumb', 'tenant'));
    }

    public function store(Request $request)
    {
        $user = auth()->user();
        $tenant = $user->company_code ?? session('company_code');
        $role = $user->role;

        // Check create permission
        if ($tenant !== '000') {
            if (!$role || !$role->hak_akses_tbl_role_cread) {
                return redirect()->route('role.index')->with('error', 'Anda tidak memiliki akses untuk menambah role.');
            }
        }

        $validated = $request->validate([
            'company_code' => 'required|exists:tbl_company,company_code',
            'nama_role' => 'required|string|max:100',
            'keterangan' => 'nullable|string',
            'hak_akses_tbl_company_cread' => 'nullable|boolean',
            'hak_akses_tbl_company_read' => 'nullable|boolean',
            'hak_akses_tbl_company_edit' => 'nullable|boolean',
            'hak_akses_tbl_company_delete' => 'nullable|boolean',
            'hak_akses_tbl_role_cread' => 'nullable|boolean',
            'hak_akses_tbl_role_read' => 'nullable|boolean',
            'hak_akses_tbl_role_edit' => 'nullable|boolean',
            'hak_akses_tbl_role_delete' => 'nullable|boolean',
            'hak_akses_tbl_pengguna_cread' => 'nullable|boolean',
            'hak_akses_tbl_pengguna_read' => 'nullable|boolean',
            'hak_akses_tbl_pengguna_edit' => 'nullable|boolean',
            'hak_akses_tbl_pengguna_delete' => 'nullable|boolean',
        ]);

        // Convert checkbox values to boolean
        foreach ($validated as $key => $value) {
            if (strpos($key, 'hak_akses_') === 0) {
                $validated[$key] = $value ?? 0;
            }
        }

        Role::create($validated);

        return redirect()->route('role.index')->with('success', 'Data berhasil ditambahkan');
    }

    public function edit($id)
    {
        $user = auth()->user();
        $tenant = $user->company_code ?? session('company_code');
        $role = $user->role;

        // Check edit permission
        if ($tenant !== '000') {
            if (!$role || !$role->hak_akses_tbl_role_edit) {
                return redirect()->route('role.index')->with('error', 'Anda tidak memiliki akses untuk mengedit role.');
            }
        }

        $roleModel = Role::findOrFail($id);

        // Ensure tenant can only edit their own rolls
        if ($tenant && $tenant !== '000' && $roleModel->company_code !== $tenant) {
            return redirect()->route('role.index')->with('error', 'Akses ditolak');
        }

        $companies = Company::when($tenant && $tenant !== '000', function ($q) use ($tenant) {
            $q->where('company_code', $tenant);
        })->get();
        $breadcrumb = [
            ['name' => 'Home', 'url' => route('dashboard.index')],
            ['name' => 'Role', 'url' => route('role.index')],
            ['name' => 'Edit', 'url' => ''],
        ];

        return view('role.edit', ['role' => $roleModel, 'companies' => $companies, 'breadcrumb' => $breadcrumb]);
    }

    public function update(Request $request, $id)
    {
        $user = auth()->user();
        $tenant = $user->company_code ?? session('company_code');
        $role = $user->role;

        // Check update permission
        if ($tenant !== '000') {
            if (!$role || !$role->hak_akses_tbl_role_edit) {
                return redirect()->route('role.index')->with('error', 'Anda tidak memiliki akses untuk mengedit role.');
            }
        }

        $roleModel = Role::findOrFail($id);

        if ($tenant && $tenant !== '000' && $roleModel->company_code !== $tenant) {
            return redirect()->route('role.index')->with('error', 'Akses ditolak');
        }

        $validated = $request->validate([
            'company_code' => 'required|exists:tbl_company,company_code',
            'nama_role' => 'required|string|max:100',
            'keterangan' => 'nullable|string',
            'hak_akses_tbl_company_cread' => 'nullable|boolean',
            'hak_akses_tbl_company_read' => 'nullable|boolean',
            'hak_akses_tbl_company_edit' => 'nullable|boolean',
            'hak_akses_tbl_company_delete' => 'nullable|boolean',
            'hak_akses_tbl_role_cread' => 'nullable|boolean',
            'hak_akses_tbl_role_read' => 'nullable|boolean',
            'hak_akses_tbl_role_edit' => 'nullable|boolean',
            'hak_akses_tbl_role_delete' => 'nullable|boolean',
            'hak_akses_tbl_pengguna_cread' => 'nullable|boolean',
            'hak_akses_tbl_pengguna_read' => 'nullable|boolean',
            'hak_akses_tbl_pengguna_edit' => 'nullable|boolean',
            'hak_akses_tbl_pengguna_delete' => 'nullable|boolean',
        ]);

        // Explicitly set all checkboxes to 0 if not present
        $checkboxes = [
            'hak_akses_tbl_company_cread',
            'hak_akses_tbl_company_read',
            'hak_akses_tbl_company_edit',
            'hak_akses_tbl_company_delete',
            'hak_akses_tbl_role_cread',
            'hak_akses_tbl_role_read',
            'hak_akses_tbl_role_edit',
            'hak_akses_tbl_role_delete',
            'hak_akses_tbl_pengguna_cread',
            'hak_akses_tbl_pengguna_read',
            'hak_akses_tbl_pengguna_edit',
            'hak_akses_tbl_pengguna_delete',
        ];

        foreach ($checkboxes as $checkbox) {
            $validated[$checkbox] = $request->has($checkbox) ? 1 : 0;
        }

        $roleModel->update($validated);

        return redirect()->route('role.index')->with('success', 'Data berhasil diperbarui');
    }

    public function destroy($id)
    {
        $user = auth()->user();
        $tenant = $user->company_code ?? session('company_code');
        $role = $user->role;

        // Check delete permission
        if ($tenant !== '000') {
            if (!$role || !$role->hak_akses_tbl_role_delete) {
                if (request()->ajax() || request()->wantsJson()) {
                    return response()->json(['success' => false, 'message' => 'Anda tidak memiliki akses untuk menghapus role.'], 403);
                }
                return redirect()->route('role.index')->with('error', 'Anda tidak memiliki akses untuk menghapus role.');
            }
        }

        $roleModel = Role::findOrFail($id);

        if ($tenant && $tenant !== '000' && $roleModel->company_code !== $tenant) {
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json(['success' => false, 'message' => 'Akses ditolak.'], 403);
            }
            return redirect()->route('role.index')->with('error', 'Akses ditolak');
        }

        $roleModel->delete();

        if (request()->ajax() || request()->wantsJson()) {
            return response()->json(['success' => true, 'message' => 'Data berhasil dihapus']);
        }

        return redirect()->route('role.index')->with('success', 'Data berhasil dihapus');
    }
}
