<?php

namespace App\Http\Controllers;

use App\Models\Product;
use Illuminate\Http\Request;

class ProductController extends Controller
{
    public function index(Request $request)
    {
        $user = auth()->user();
        $tenant = $user?->company_code ?? session('company_code');

        $limit = $request->input('limit', 40);
        $search = $request->input('search');

        $products = Product::query()
            ->when($tenant && $tenant !== '000', function ($q) use ($tenant) {
                $q->where('company_code', $tenant);
            })
            ->when($search, function ($query, $search) {
                return $query->where('nama', 'like', "%{$search}%")
                    ->orWhere('price', 'like', "%{$search}%");
            })
            ->paginate($limit)
            ->withQueryString();

        $breadcrumb = [
            ['name' => 'Home', 'url' => route('dashboard.index')],
            ['name' => 'Product', 'url' => ''],
        ];

        return view('product.index', compact('products', 'breadcrumb'));
    }

    public function create()
    {
        $breadcrumb = [
            ['name' => 'Home', 'url' => route('dashboard.index')],
            ['name' => 'Product', 'url' => route('product.index')],
            ['name' => 'Tambah', 'url' => ''],
        ];

        return view('product.create', compact('breadcrumb'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'nama' => 'required|string|max:255',
            'price' => 'required|numeric|min:0',
        ]);

        Product::create($validated);

        return redirect()->route('product.index')->with('success', 'Data berhasil ditambahkan');
    }

    public function edit($id)
    {
        $user = auth()->user();
        $tenant = $user?->company_code ?? session('company_code');

        $product = Product::findOrFail($id);

        if ($tenant && $tenant !== '000' && $product->company_code !== $tenant) {
            return redirect()->route('product.index')->with('error', 'Akses ditolak');
        }

        $breadcrumb = [
            ['name' => 'Home', 'url' => route('dashboard.index')],
            ['name' => 'Product', 'url' => route('product.index')],
            ['name' => 'Edit', 'url' => ''],
        ];

        return view('product.edit', compact('product', 'breadcrumb'));
    }

    public function update(Request $request, $id)
    {
        $user = auth()->user();
        $tenant = $user?->company_code ?? session('company_code');

        $product = Product::findOrFail($id);

        if ($tenant && $tenant !== '000' && $product->company_code !== $tenant) {
            return redirect()->route('product.index')->with('error', 'Akses ditolak');
        }

        $validated = $request->validate([
            'nama' => 'required|string|max:255',
            'price' => 'required|numeric|min:0',
        ]);

        $product->update($validated);

        return redirect()->route('product.index')->with('success', 'Data berhasil diperbarui');
    }

    public function destroy($id)
    {
        $user = auth()->user();
        $tenant = $user?->company_code ?? session('company_code');

        $product = Product::findOrFail($id);

        if ($tenant && $tenant !== '000' && $product->company_code !== $tenant) {
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json(['success' => false, 'message' => 'Akses ditolak'], 403);
            }
            return redirect()->route('product.index')->with('error', 'Akses ditolak');
        }

        $product->delete();

        if (request()->ajax() || request()->wantsJson()) {
            return response()->json(['success' => true, 'message' => 'Data berhasil dihapus']);
        }

        return redirect()->route('product.index')->with('success', 'Data berhasil dihapus');
    }
}
