<?php

namespace App\Http\Controllers;

use App\Models\Pengguna;
use App\Models\Role;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class PenggunaController extends Controller
{
    public function index(Request $request)
    {
        $user = auth()->user();
        $tenant = $user?->company_code ?? session('company_code');
        $role = $user?->role;

        // Check read permission (Superadmin 000 bypasses check)
        if ($tenant !== '000') {
            if (!$role || !$role->hak_akses_tbl_pengguna_read) {
                abort(403, 'Anda tidak memiliki akses untuk melihat data pengguna.');
            }
        }

        $perPage = $request->get('per_page', 40);
        $search  = $request->get('search');

        $query = Pengguna::with(['role', 'company']); // Eager load company
        if ($tenant !== '000' && $tenant) {
            $query->where('company_code', $tenant);
        }

        if ($search) {
            $query->where('nama', 'like', "%{$search}%")
                ->orWhere('username', 'like', "%{$search}%")
                ->orWhere('email', 'like', "%{$search}%")
                ->orWhere('hp', 'like', "%{$search}%")
                ->orWhereHas('role', function ($q) use ($search) {
                    $q->where('nama_role', 'like', "%{$search}%");
                });
        }

        // Sort: superadmin (000) first, then alphabetically by company_code, then by nama
        $penggunas = $query
            ->orderByRaw("CASE WHEN company_code = '000' THEN 0 ELSE 1 END")
            ->orderBy('company_code', 'asc')
            ->orderBy('nama', 'asc')
            ->paginate($perPage)
            ->withQueryString();

        // Permission Checks
        if ($tenant === '000') {
            $canCreate = true;
            $canEdit = true;
            $canDelete = true;
        } else {
            $canCreate = ($role && $role->hak_akses_tbl_pengguna_cread);
            $canEdit = ($role && $role->hak_akses_tbl_pengguna_edit);
            $canDelete = ($role && $role->hak_akses_tbl_pengguna_delete);
        }

        $breadcrumb = [
            ['name' => 'Dashboard', 'url' => route('dashboard.index')],
            ['name' => 'Pengguna', 'url' => route('pengguna.index')],
        ];

        return view('pengguna.index', compact('penggunas', 'breadcrumb', 'search', 'perPage', 'canCreate', 'canEdit', 'canDelete'));
    }

    public function create()
    {
        $user = auth()->user();
        $tenant = $user->company_code ?? session('company_code');
        $role = $user->role;

        // Check create permission
        if ($tenant !== '000') {
            if (!$role || !$role->hak_akses_tbl_pengguna_cread) {
                return redirect()->route('pengguna.index')->with('error', 'Anda tidak memiliki akses untuk menambah pengguna.');
            }
        }

        $userCompanyCode = $tenant; // Pass user's company_code to view

        $rolesQ = Role::orderBy('nama_role');
        if ($tenant !== '000' && $tenant) {
            $rolesQ->where('company_code', $tenant);
        }
        $roles = $rolesQ->pluck('nama_role', 'id');
        $companiesQ = \App\Models\Company::select('company_code', 'nama_company')->orderBy('company_code');
        if ($tenant !== '000' && $tenant) {
            $companiesQ->where('company_code', $tenant);
        }
        $companies = $companiesQ->get();

        $breadcrumb = [
            ['name' => 'Dashboard', 'url' => route('dashboard.index')],
            ['name' => 'Pengguna', 'url' => route('pengguna.index')],
            ['name' => 'Tambah Pengguna', 'url' => ''],
        ];

        return view('pengguna.create', compact('breadcrumb', 'roles', 'companies', 'userCompanyCode'));
    }

    public function store(Request $request)
    {
        $user = auth()->user();
        $tenant = $user->company_code ?? session('company_code');
        $role = $user->role;

        // Check create permission
        if ($tenant !== '000') {
            if (!$role || !$role->hak_akses_tbl_pengguna_cread) {
                return redirect()->route('pengguna.index')->with('error', 'Anda tidak memiliki akses untuk menambah pengguna.');
            }
        }

        $request->validate([
            'company_code' => 'required|exists:tbl_company,company_code',
            'nama'         => 'required|string|max:100',
            'username'     => 'required|string|max:50|unique:tbl_pengguna,username',
            'email'        => 'nullable|email|max:100|unique:tbl_pengguna,email',
            'password'     => 'required|string|min:6|confirmed',
            'roleid'       => 'required|exists:tbl_role,id',
            'hp'           => 'nullable|string|max:20',
            'keterangan'   => 'nullable|string',
        ]);

        Pengguna::create([
            'company_code' => $request->company_code,
            'nama'         => $request->nama,
            'username'     => $request->username,
            'email'        => $request->email,
            'password'     => Hash::make($request->password),
            'roleid'       => $request->roleid,
            'hp'           => $request->hp,
            'keterangan'   => $request->keterangan,
        ]);

        return redirect()
            ->route('pengguna.index')
            ->with('success', 'Pengguna berhasil ditambahkan');
    }

    public function edit(Pengguna $pengguna)
    {
        $user = auth()->user();
        $tenant = $user->company_code ?? session('company_code');
        $role = $user->role;

        // Check edit permission
        if ($tenant !== '000') {
            if (!$role || !$role->hak_akses_tbl_pengguna_edit) {
                return redirect()->route('pengguna.index')->with('error', 'Anda tidak memiliki akses untuk mengedit pengguna.');
            }
        }

        // Ensure tenant can only edit their own users (unless 000)
        if ($tenant && $tenant !== '000' && $pengguna->company_code !== $tenant) {
            return redirect()->route('pengguna.index')->with('error', 'Akses ditolak');
        }

        $rolesQ = Role::orderBy('nama_role');
        if ($tenant !== '000' && $tenant) {
            $rolesQ->where('company_code', $tenant);
        }
        $roles = $rolesQ->pluck('nama_role', 'id');
        $companiesQ = \App\Models\Company::select('company_code', 'nama_company')->orderBy('company_code');
        if ($tenant !== '000' && $tenant) {
            $companiesQ->where('company_code', $tenant);
        }
        $companies = $companiesQ->get();

        $breadcrumb = [
            ['name' => 'Dashboard', 'url' => route('dashboard.index')],
            ['name' => 'Pengguna', 'url' => route('pengguna.index')],
            ['name' => 'Edit Pengguna', 'url' => ''],
        ];

        return view('pengguna.edit', compact('pengguna', 'breadcrumb', 'roles', 'companies'));
    }

    public function update(Request $request, Pengguna $pengguna)
    {
        $user = auth()->user();
        $tenant = $user->company_code ?? session('company_code');
        $role = $user->role;

        // Check update permission
        if ($tenant !== '000') {
            if (!$role || !$role->hak_akses_tbl_pengguna_edit) {
                return redirect()->route('pengguna.index')->with('error', 'Anda tidak memiliki akses untuk mengedit pengguna.');
            }
        }

        // Ensure tenant can only edit their own users (unless 000)
        if ($tenant && $tenant !== '000' && $pengguna->company_code !== $tenant) {
            return redirect()->route('pengguna.index')->with('error', 'Akses ditolak');
        }

        $request->validate([
            'company_code' => 'required|exists:tbl_company,company_code',
            'nama'         => 'required|string|max:100',
            'username'     => 'required|string|max:50|unique:tbl_pengguna,username,' . $pengguna->id,
            'email'        => 'nullable|email|max:100|unique:tbl_pengguna,email,' . $pengguna->id,
            'password'     => 'nullable|string|min:6|confirmed',
            'roleid'       => 'required|exists:tbl_role,id',
            'hp'           => 'nullable|string|max:20',
            'keterangan'   => 'nullable|string',
        ]);

        $data = $request->only([
            'company_code',
            'nama',
            'username',
            'email',
            'roleid',
            'hp',
            'keterangan'
        ]);

        if ($request->filled('password')) {
            $data['password'] = Hash::make($request->password);
        }

        $pengguna->update($data);

        return redirect()
            ->route('pengguna.index')
            ->with('success', 'Pengguna berhasil diperbarui');
    }

    public function destroy(Pengguna $pengguna)
    {
        $user = auth()->user();
        $tenant = $user->company_code ?? session('company_code');
        $role = $user->role;

        // Check delete permission
        if ($tenant !== '000') {
            if (!$role || !$role->hak_akses_tbl_pengguna_delete) {
                if (request()->ajax() || request()->wantsJson()) {
                    return response()->json(['success' => false, 'message' => 'Anda tidak memiliki akses untuk menghapus pengguna.'], 403);
                }
                return redirect()->route('pengguna.index')->with('error', 'Anda tidak memiliki akses untuk menghapus pengguna.');
            }
        }

        // Ensure tenant can only delete their own users (unless 000)
        if ($tenant && $tenant !== '000' && $pengguna->company_code !== $tenant) {
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json(['success' => false, 'message' => 'Akses ditolak.'], 403);
            }
            return redirect()->route('pengguna.index')->with('error', 'Akses ditolak');
        }

        $pengguna->delete();

        // Kalau request dari AJAX (SweetAlert2 fetch) → return JSON
        if (request()->ajax() || request()->wantsJson()) {
            return response()->json(['success' => true, 'message' => 'Pengguna berhasil dihapus']);
        }

        // Kalau form biasa → redirect + flash message
        return redirect()
            ->route('pengguna.index')
            ->with('success', 'Pengguna berhasil dihapus');
    }
}
