<?php

namespace App\Http\Controllers;

use App\Models\PelaporanGangguan;
use App\Models\Pelanggan;
use App\Models\Pengguna;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class PelaporanGangguanController extends Controller
{
    public function index(Request $request)
    {
        $user = Auth::user();
        $tenant = $user?->company_code ?? session('company_code');

        $limit = $request->input('limit', 40);
        $search = $request->input('search');

        $pelaporanGangguans = PelaporanGangguan::when($tenant && $tenant !== '000', function ($q) use ($tenant) {
            $q->where('company_code', $tenant);
        })
            ->when($search, function ($query, $search) {
                return $query->where(function ($q) use ($search) {
                    $q->where('ticket', 'like', "%{$search}%")
                        ->orWhere('nama', 'like', "%{$search}%")
                        ->orWhere('alamat', 'like', "%{$search}%")
                        ->orWhere('hp', 'like', "%{$search}%")
                        ->orWhere('jenis_gangguan', 'like', "%{$search}%");
                });
            })
            ->orderBy('created_at', 'desc')
            ->paginate($limit)
            ->withQueryString();

        $breadcrumb = [
            ['name' => 'Home', 'url' => route('dashboard.index')],
            ['name' => 'Pelaporan Gangguan', 'url' => ''],
        ];

        return view('pelaporangangguan.index', compact('pelaporanGangguans', 'breadcrumb'));
    }

    public function create()
    {
        $user = Auth::user();
        $tenant = $user?->company_code ?? session('company_code');

        // Get pelanggan data for selection
        $pelanggans = Pelanggan::when($tenant && $tenant !== '000', function ($q) use ($tenant) {
            $q->where('company_code', $tenant);
        })->get();

        // Get teknisi data for selection
        $teknisis = Pengguna::when($tenant && $tenant !== '000', function ($q) use ($tenant) {
            $q->where('company_code', $tenant);
        })->get();

        // Get pengguna data for penerima laporan
        $penerimaLaporans = Pengguna::when($tenant && $tenant !== '000', function ($q) use ($tenant) {
            $q->where('company_code', $tenant);
        })->get();

        $breadcrumb = [
            ['name' => 'Home', 'url' => route('dashboard.index')],
            ['name' => 'Pelaporan Gangguan', 'url' => route('pelaporangangguan.index')],
            ['name' => 'Tambah', 'url' => ''],
        ];

        return view('pelaporangangguan.create', compact('pelanggans', 'teknisis', 'penerimaLaporans', 'breadcrumb'));
    }

    public function store(Request $request)
    {
        $user = Auth::user();
        $tenant = $user?->company_code ?? session('company_code');

        $validated = $request->validate([
            'nama' => 'required|string|max:150',
            'alamat' => 'nullable|string',
            'hp' => 'nullable|string|max:30',
            'jenis_gangguan' => 'nullable|string|max:200',
            'jam_ganguan' => 'nullable|date',
            'penerima_laporan' => 'required|string|max:150',
            'waktu_mulai_penanganan' => 'nullable|date',
            'petugas_teknisi' => 'required|string|max:150',
            'waktu_finish' => 'nullable|date',
            'foto1' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:10240',
            'foto2' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:10240',
            'keterangan' => 'nullable|string',
        ]);

        // Handle file uploads for foto1
        if ($request->hasFile('foto1')) {
            $validated['foto1'] = $this->handleImageUpload($request->file('foto1'), 'pelaporan_gangguan_fotos');
        }

        // Handle file uploads for foto2
        if ($request->hasFile('foto2')) {
            $validated['foto2'] = $this->handleImageUpload($request->file('foto2'), 'pelaporan_gangguan_fotos');
        }

        // Generate ticket
        $validated['ticket'] = $this->generateTicket($tenant);

        // Format dates to ensure MySQL compatibility
        if (!empty($validated['jam_ganguan'])) {
            $validated['jam_ganguan'] = str_replace('T', ' ', $validated['jam_ganguan']);
        }
        if (!empty($validated['waktu_mulai_penanganan'])) {
            $validated['waktu_mulai_penanganan'] = str_replace('T', ' ', $validated['waktu_mulai_penanganan']);
        }
        if (!empty($validated['waktu_finish'])) {
            $validated['waktu_finish'] = str_replace('T', ' ', $validated['waktu_finish']);
        }

        PelaporanGangguan::create($validated);

        return redirect()->route('pelaporangangguan.index')->with('success', 'Data pelaporan gangguan berhasil ditambahkan');
    }

    public function show($id)
    {
        $user = Auth::user();
        $tenant = $user?->company_code ?? session('company_code');

        $pelaporanGangguan = PelaporanGangguan::when($tenant && $tenant !== '000', function ($q) use ($tenant) {
            $q->where('company_code', $tenant);
        })->findOrFail($id);

        $breadcrumb = [
            ['name' => 'Home', 'url' => route('dashboard.index')],
            ['name' => 'Pelaporan Gangguan', 'url' => route('pelaporangangguan.index')],
            ['name' => 'Detail', 'url' => ''],
        ];

        return view('pelaporangangguan.show', compact('pelaporanGangguan', 'breadcrumb'));
    }

    public function edit($id)
    {
        $user = Auth::user();
        $tenant = $user?->company_code ?? session('company_code');

        $pelaporanGangguan = PelaporanGangguan::when($tenant && $tenant !== '000', function ($q) use ($tenant) {
            $q->where('company_code', $tenant);
        })->findOrFail($id);

        // Get pelanggan data for selection
        $pelanggans = Pelanggan::when($tenant && $tenant !== '000', function ($q) use ($tenant) {
            $q->where('company_code', $tenant);
        })->get();

        // Get teknisi data for selection
        $teknisis = Pengguna::when($tenant && $tenant !== '000', function ($q) use ($tenant) {
            $q->where('company_code', $tenant);
        })->get();

        // Get pengguna data for penerima laporan
        $penerimaLaporans = Pengguna::when($tenant && $tenant !== '000', function ($q) use ($tenant) {
            $q->where('company_code', $tenant);
        })->get();

        $breadcrumb = [
            ['name' => 'Home', 'url' => route('dashboard.index')],
            ['name' => 'Pelaporan Gangguan', 'url' => route('pelaporangangguan.index')],
            ['name' => 'Edit', 'url' => ''],
        ];

        return view('pelaporangangguan.edit', compact('pelaporanGangguan', 'pelanggans', 'teknisis', 'penerimaLaporans', 'breadcrumb'));
    }

    public function update(Request $request, $id)
    {
        $user = Auth::user();
        $tenant = $user?->company_code ?? session('company_code');

        $pelaporanGangguan = PelaporanGangguan::when($tenant && $tenant !== '000', function ($q) use ($tenant) {
            $q->where('company_code', $tenant);
        })->findOrFail($id);

        $validated = $request->validate([
            'nama' => 'sometimes|required|string|max:150',
            'alamat' => 'nullable|string',
            'hp' => 'nullable|string|max:30',
            'jenis_gangguan' => 'nullable|string|max:200',
            'jam_ganguan' => 'nullable|date',
            'penerima_laporan' => 'nullable|string|max:150',
            'waktu_mulai_penanganan' => 'nullable|date',
            'petugas_teknisi' => 'nullable|string|max:150',
            'waktu_finish' => 'nullable|date',
            'foto1' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:10240',
            'foto2' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:10240',
            'keterangan' => 'nullable|string',
        ]);

        // Handle file uploads for foto1
        if ($request->hasFile('foto1')) {
            // Delete old foto1 if exists
            if ($pelaporanGangguan->foto1) {
                Storage::disk('public')->delete($pelaporanGangguan->foto1);
            }

            $validated['foto1'] = $this->handleImageUpload($request->file('foto1'), 'pelaporan_gangguan_fotos');
        }

        // Handle file uploads for foto2
        if ($request->hasFile('foto2')) {
            // Delete old foto2 if exists
            if ($pelaporanGangguan->foto2) {
                Storage::disk('public')->delete($pelaporanGangguan->foto2);
            }

            $validated['foto2'] = $this->handleImageUpload($request->file('foto2'), 'pelaporan_gangguan_fotos');
        }

        // Format dates to ensure MySQL compatibility
        if (!empty($validated['jam_ganguan'])) {
            $validated['jam_ganguan'] = str_replace('T', ' ', $validated['jam_ganguan']);
        }
        if (!empty($validated['waktu_mulai_penanganan'])) {
            $validated['waktu_mulai_penanganan'] = str_replace('T', ' ', $validated['waktu_mulai_penanganan']);
        }
        if (!empty($validated['waktu_finish'])) {
            $validated['waktu_finish'] = str_replace('T', ' ', $validated['waktu_finish']);
        }

        $pelaporanGangguan->update($validated);

        return redirect()->route('pelaporangangguan.index')->with('success', 'Data pelaporan gangguan berhasil diperbarui');
    }

    public function destroy($id)
    {
        $user = Auth::user();
        $tenant = $user?->company_code ?? session('company_code');

        $pelaporanGangguan = PelaporanGangguan::when($tenant && $tenant !== '000', function ($q) use ($tenant) {
            $q->where('company_code', $tenant);
        })->findOrFail($id);

        // Delete foto files if they exist
        if ($pelaporanGangguan->foto1) {
            Storage::disk('public')->delete($pelaporanGangguan->foto1);
        }

        if ($pelaporanGangguan->foto2) {
            Storage::disk('public')->delete($pelaporanGangguan->foto2);
        }

        $pelaporanGangguan->delete();

        if (request()->ajax() || request()->wantsJson()) {
            return response()->json(['success' => true, 'message' => 'Data pelaporan gangguan berhasil dihapus']);
        }

        return redirect()->route('pelaporangangguan.index')->with('success', 'Data pelaporan gangguan berhasil dihapus');
    }

    public function finish($id)
    {
        $user = Auth::user();
        $tenant = $user?->company_code ?? session('company_code');

        $pelaporanGangguan = PelaporanGangguan::when($tenant && $tenant !== '000', function ($q) use ($tenant) {
            $q->where('company_code', $tenant);
        })->findOrFail($id);

        $pelaporanGangguan->update(['finish' => 1]);

        return redirect()->route('pelaporangangguan.index')->with('success', 'Status pelaporan gangguan berhasil diupdate menjadi finish');
    }

    private function generateTicket($companyCode)
    {
        $date = now()->format('dmy'); // ddmmyy format
        $suffix = "-{$date}-{$companyCode}";

        // Find the last ticket that matches the pattern for today and this company
        $lastTicket = PelaporanGangguan::where('ticket', 'like', "%{$suffix}")
            ->orderBy('id', 'desc')
            ->first();

        $noUrut = 1;

        if ($lastTicket) {
            $parts = explode('-', $lastTicket->ticket);
            // Check if the first part is numeric and increment
            if (is_numeric($parts[0])) {
                $noUrut = intval($parts[0]) + 1;
            }
        }

        return sprintf('%05d-%s-%s', $noUrut, $date, $companyCode);
    }

    private function handleImageUpload($file, $path)
    {
        // If file <= 1MB, store normally
        if ($file->getSize() <= 1048576) {
            return $file->store($path, 'public');
        }

        // Attempt compression
        try {
            $sourceImage = imagecreatefromstring(file_get_contents($file->getRealPath()));
            if (!$sourceImage) {
                return $file->store($path, 'public');
            }

            // Get dimensions
            $width = imagesx($sourceImage);
            $height = imagesy($sourceImage);

            // Resize if too big (width > 1280) to help achieve target size
            $targetWidth = 1280;
            if ($width > $targetWidth) {
                $targetHeight = ($height / $width) * $targetWidth;
                $newImage = imagecreatetruecolor($targetWidth, $targetHeight);

                // Handle transprency by filling white background (since we convert to JPEG)
                $white = imagecolorallocate($newImage, 255, 255, 255);
                imagefilledrectangle($newImage, 0, 0, $targetWidth, $targetHeight, $white);

                imagecopyresampled($newImage, $sourceImage, 0, 0, 0, 0, $targetWidth, $targetHeight, $width, $height);
                imagedestroy($sourceImage); // Free old memory
                $sourceImage = $newImage;
            }

            // Compress to temp file
            $tempFile = tempnam(sys_get_temp_dir(), 'img_cmp');

            // Start with quality 80
            imagejpeg($sourceImage, $tempFile, 80);

            // If still > 1MB, reduce quality
            $quality = 80;
            while (filesize($tempFile) > 1048576 && $quality > 10) {
                $quality -= 10;
                imagejpeg($sourceImage, $tempFile, $quality);
            }

            // Save to public storage
            $filename = $file->hashName();
            // Force .jpg extension since we converted to JPEG
            $filename = pathinfo($filename, PATHINFO_FILENAME) . '.jpg';
            $fullPath = $path . '/' . $filename;

            Storage::disk('public')->put($fullPath, file_get_contents($tempFile));

            // Cleanup
            imagedestroy($sourceImage);
            unlink($tempFile);

            return $fullPath;
        } catch (\Exception $e) {
            // Fallback in case of any error
            return $file->store($path, 'public');
        }
    }
}
