<?php

namespace App\Http\Controllers;

use App\Models\PasangBaru;
use App\Models\Pelanggan;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class PasangBaruController extends Controller
{
    public function index(Request $request)
    {
        $user = Auth::user();
        $tenant = $user?->company_code ?? session('company_code');

        $limit = $request->input('limit', 40);
        $search = $request->input('search');

        $pasangbarus = PasangBaru::with('company', 'product')
            ->when($tenant && $tenant !== '000', function ($q) use ($tenant) {
                $q->where('company_code', $tenant);
            })
            ->when($search, function ($query, $search) {
                return $query->where('nama', 'like', "%{$search}%")
                    ->orWhere('alamat', 'like', "%{$search}%")
                    ->orWhere('hp', 'like', "%{$search}%")
                    ->orWhere('username_pppoe', 'like', "%{$search}%")
                    ->orWhereHas('company', function ($q) use ($search) {
                        $q->where('nama_company', 'like', "%{$search}%");
                    });
            })
            ->orderBy('nama')
            ->paginate($limit)
            ->withQueryString();

        $breadcrumb = [
            ['name' => 'Home', 'url' => route('dashboard.index')],
            ['name' => 'Pasang Baru', 'url' => ''],
        ];

        return view('pasangbaru.index', compact('pasangbarus', 'breadcrumb'));
    }

    public function create()
    {
        $user = Auth::user();
        $tenant = $user?->company_code ?? session('company_code');

        // Filter products based on tenant
        $products = Product::when($tenant && $tenant !== '000', function ($q) use ($tenant) {
            $q->where('company_code', $tenant);
        })->get();

        $breadcrumb = [
            ['name' => 'Home', 'url' => route('dashboard.index')],
            ['name' => 'Pasang Baru', 'url' => route('pasangbaru.index')],
            ['name' => 'Tambah', 'url' => ''],
        ];

        return view('pasangbaru.create', compact('products', 'breadcrumb'));
    }

    public function store(Request $request)
    {
        $user = Auth::user();
        $tenant = $user?->company_code ?? session('company_code');

        $validated = $request->validate([
            'nama' => 'required|string|max:150',
            'alamat' => 'required|string',
            'hp' => 'required|string|max:20',
            'paket_product' => 'required|exists:tbl_product,id',
            'username_pppoe' => 'nullable|string|max:100',
            'password' => 'nullable|string|max:100',
            'tgl_pendaftaran' => 'required|date',
            'tgl_pasang' => 'nullable|date',
            'tagihan_dimulai_tgl' => 'nullable|date',
            'map' => 'nullable|string|max:255',
            'foto1' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:10240',
            'foto2' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:10240',
            'status' => 'required|in:active,inactive',
            'keterangan' => 'nullable|string',
            'finish' => 'nullable|string|in:sudah,belum',
        ]);

        // Handle file uploads for foto1
        if ($request->hasFile('foto1')) {
            $validated['foto1'] = $this->handleImageUpload($request->file('foto1'), 'pasangbaru_fotos');
        }

        // Handle file uploads for foto2
        if ($request->hasFile('foto2')) {
            $validated['foto2'] = $this->handleImageUpload($request->file('foto2'), 'pasangbaru_fotos');
        }

        // Set company_code from authenticated user
        $validated['company_code'] = $tenant;

        PasangBaru::create($validated);

        return redirect()->route('pasangbaru.index')->with('success', 'Data pasang baru berhasil ditambahkan');
    }

    public function show($id)
    {
        $user = Auth::user();
        $tenant = $user?->company_code ?? session('company_code');

        $pasangbaru = PasangBaru::with('company', 'product')->findOrFail($id);

        if ($tenant && $tenant !== '000' && $pasangbaru->company_code !== $tenant) {
            return redirect()->route('pasangbaru.index')->with('error', 'Akses ditolak');
        }

        $breadcrumb = [
            ['name' => 'Home', 'url' => route('dashboard.index')],
            ['name' => 'Pasang Baru', 'url' => route('pasangbaru.index')],
            ['name' => 'Detail', 'url' => ''],
        ];

        return view('pasangbaru.show', compact('pasangbaru', 'breadcrumb'));
    }

    public function edit($id)
    {
        $user = Auth::user();
        $tenant = $user?->company_code ?? session('company_code');

        $pasangbaru = PasangBaru::with('company', 'product')->findOrFail($id);

        if ($tenant && $tenant !== '000' && $pasangbaru->company_code !== $tenant) {
            return redirect()->route('pasangbaru.index')->with('error', 'Akses ditolak');
        }

        // Filter products based on tenant
        $products = Product::when($tenant && $tenant !== '000', function ($q) use ($tenant) {
            $q->where('company_code', $tenant);
        })->get();

        $breadcrumb = [
            ['name' => 'Home', 'url' => route('dashboard.index')],
            ['name' => 'Pasang Baru', 'url' => route('pasangbaru.index')],
            ['name' => 'Edit', 'url' => ''],
        ];

        return view('pasangbaru.edit', compact('pasangbaru', 'products', 'breadcrumb'));
    }

    public function update(Request $request, $id)
    {
        $user = Auth::user();
        $tenant = $user?->company_code ?? session('company_code');

        $pasangbaru = PasangBaru::findOrFail($id);

        if ($tenant && $tenant !== '000' && $pasangbaru->company_code !== $tenant) {
            return redirect()->route('pasangbaru.index')->with('error', 'Akses ditolak');
        }

        $validated = $request->validate([
            'nama' => 'required|string|max:150',
            'alamat' => 'required|string',
            'hp' => 'required|string|max:20',
            'paket_product' => 'required|exists:tbl_product,id',
            'username_pppoe' => 'nullable|string|max:100',
            'password' => 'nullable|string|max:100',
            'tgl_pendaftaran' => 'required|date',
            'tgl_pasang' => 'nullable|date',
            'tagihan_dimulai_tgl' => 'nullable|date',
            'map' => 'nullable|string|max:255',
            'foto1' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:10240',
            'foto2' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:10240',
            'status' => 'required|in:active,inactive',
            'keterangan' => 'nullable|string',
            'finish' => 'nullable|string|in:sudah,belum',
        ]);

        // Handle file uploads for foto1
        if ($request->hasFile('foto1')) {
            // Delete old foto1 if exists
            if ($pasangbaru->foto1) {
                Storage::disk('public')->delete($pasangbaru->foto1);
            }

            $validated['foto1'] = $this->handleImageUpload($request->file('foto1'), 'pasangbaru_fotos');
        }

        // Handle file uploads for foto2
        if ($request->hasFile('foto2')) {
            // Delete old foto2 if exists
            if ($pasangbaru->foto2) {
                Storage::disk('public')->delete($pasangbaru->foto2);
            }

            $validated['foto2'] = $this->handleImageUpload($request->file('foto2'), 'pasangbaru_fotos');
        }

        $pasangbaru->update($validated);

        return redirect()->route('pasangbaru.index')->with('success', 'Data pasang baru berhasil diperbarui');
    }

    public function finish($id)
    {
        $user = Auth::user();
        $tenant = $user?->company_code ?? session('company_code');

        $pasangbaru = PasangBaru::findOrFail($id);

        if ($tenant && $tenant !== '000' && $pasangbaru->company_code !== $tenant) {
            return redirect()->route('pasangbaru.index')->with('error', 'Akses ditolak');
        }

        // Update the finish field to 'sudah'
        $pasangbaru->update(['finish' => 'sudah']);

        // Copy data to Pelanggan model (excluding id and finish fields)
        $attributes = $pasangbaru->getAttributes();
        $pelangganData = [];

        // Copy all attributes except id, finish, and timestamps
        foreach ($attributes as $key => $value) {
            if (!in_array($key, ['id', 'finish', 'created_at', 'updated_at'])) {
                // Handle date fields properly
                if (in_array($key, ['tgl_pendaftaran', 'tgl_pasang', 'tagihan_dimulai_tgl'])) {
                    // Convert date objects to string format or keep as null
                    if ($value instanceof \Carbon\Carbon) {
                        $pelangganData[$key] = $value->toDateString();
                    } elseif ($value instanceof \DateTime) {
                        $pelangganData[$key] = $value->format('Y-m-d');
                    } elseif (is_string($value) && !empty($value)) {
                        $pelangganData[$key] = $value;
                    } else {
                        $pelangganData[$key] = null;
                    }
                } else {
                    $pelangganData[$key] = $value;
                }
            }
        }

        // Ensure tagihan_dimulai_tgl is not null - use default logic
        if (empty($pelangganData['tagihan_dimulai_tgl'])) {
            // Default to tgl_pendaftaran if available, otherwise tgl_pasang
            if (!empty($pelangganData['tgl_pendaftaran'])) {
                $pelangganData['tagihan_dimulai_tgl'] = $pelangganData['tgl_pendaftaran'];
            } elseif (!empty($pelangganData['tgl_pasang'])) {
                $pelangganData['tagihan_dimulai_tgl'] = $pelangganData['tgl_pasang'];
            }
        }

        // Create new Pelanggan record
        Pelanggan::create($pelangganData);

        return redirect()->route('pasangbaru.index')->with('success', 'Data telah diselesaikan dan dipindahkan ke pelanggan');
    }

    public function destroy($id)
    {
        $user = Auth::user();
        $tenant = $user?->company_code ?? session('company_code');

        $pasangbaru = PasangBaru::findOrFail($id);

        if ($tenant && $tenant !== '000' && $pasangbaru->company_code !== $tenant) {
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json(['success' => false, 'message' => 'Akses ditolak'], 403);
            }
            return redirect()->route('pasangbaru.index')->with('error', 'Akses ditolak');
        }

        // Delete foto files if they exist
        if ($pasangbaru->foto1) {
            Storage::disk('public')->delete($pasangbaru->foto1);
        }

        if ($pasangbaru->foto2) {
            Storage::disk('public')->delete($pasangbaru->foto2);
        }

        $pasangbaru->delete();

        if (request()->ajax() || request()->wantsJson()) {
            return response()->json(['success' => true, 'message' => 'Data pasang baru berhasil dihapus']);
        }

        return redirect()->route('pasangbaru.index')->with('success', 'Data pasang baru berhasil dihapus');
    }

    private function handleImageUpload($file, $path)
    {
        // If file <= 1MB, store normally
        if ($file->getSize() <= 1048576) {
            return $file->store($path, 'public');
        }

        // Attempt compression
        try {
            $sourceImage = imagecreatefromstring(file_get_contents($file->getRealPath()));
            if (!$sourceImage) {
                return $file->store($path, 'public');
            }

            // Get dimensions
            $width = imagesx($sourceImage);
            $height = imagesy($sourceImage);

            // Resize if too big (width > 1280) to help achieve target size
            $targetWidth = 1280;
            if ($width > $targetWidth) {
                $targetHeight = ($height / $width) * $targetWidth;
                $newImage = imagecreatetruecolor($targetWidth, $targetHeight);

                // Handle transprency by filling white background (since we convert to JPEG)
                $white = imagecolorallocate($newImage, 255, 255, 255);
                imagefilledrectangle($newImage, 0, 0, $targetWidth, $targetHeight, $white);

                imagecopyresampled($newImage, $sourceImage, 0, 0, 0, 0, $targetWidth, $targetHeight, $width, $height);
                imagedestroy($sourceImage); // Free old memory
                $sourceImage = $newImage;
            }

            // Compress to temp file
            $tempFile = tempnam(sys_get_temp_dir(), 'img_cmp');

            // Start with quality 80
            imagejpeg($sourceImage, $tempFile, 80);

            // If still > 1MB, reduce quality
            $quality = 80;
            while (filesize($tempFile) > 1048576 && $quality > 10) {
                $quality -= 10;
                imagejpeg($sourceImage, $tempFile, $quality);
            }

            // Save to public storage
            $filename = $file->hashName();
            // Force .jpg extension since we converted to JPEG
            $filename = pathinfo($filename, PATHINFO_FILENAME) . '.jpg';
            $fullPath = $path . '/' . $filename;

            Storage::disk('public')->put($fullPath, file_get_contents($tempFile));

            // Cleanup
            imagedestroy($sourceImage);
            unlink($tempFile);

            return $fullPath;
        } catch (\Exception $e) {
            // Fallback in case of any error
            return $file->store($path, 'public');
        }
    }
}
