<?php

namespace App\Http\Controllers;

use App\Models\MetodePembayaran;
use Illuminate\Http\Request;

class MetodePembayaranController extends Controller
{
    public function index(Request $request)
    {
        $limit = $request->input('limit', 40);
        $search = $request->input('search');
        $user = auth()->user();
        $tenant = $user->company_code ?? session('company_code');

        $query = MetodePembayaran::with('company');

        // Isolation: Superadmin (000) sees all, others see only their company
        if ($tenant !== '000') {
            $query->where('company_code', $tenant);
        }

        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('metode_pembayaran', 'like', "%{$search}%")
                    ->orWhere('keterangan', 'like', "%{$search}%")
                    ->orWhere('company_code', 'like', "%{$search}%");
            });
        }

        $metodePembayarans = $query->latest()->paginate($limit)->withQueryString();

        $breadcrumb = [
            ['name' => 'Home', 'url' => route('dashboard.index')],
            ['name' => 'Metode Pembayaran', 'url' => route('metodepembayaran.index')],
        ];

        return view('metodepembayaran.index', compact('metodePembayarans', 'breadcrumb'));
    }

    public function create()
    {
        $breadcrumb = [
            ['name' => 'Home', 'url' => route('dashboard.index')],
            ['name' => 'Metode Pembayaran', 'url' => route('metodepembayaran.index')],
            ['name' => 'Tambah', 'url' => ''],
        ];

        return view('metodepembayaran.create', compact('breadcrumb'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'metode_pembayaran' => 'required|string|max:100',
            'keterangan' => 'nullable|string',
        ]);

        // company_code is handled by model boot method, or we can explicit it here if needed
        // but user requirement said "automatically filled according to Model Company" 
        // which we implemented in the model boot.

        // However, for superadmin '000', they might want to assign it? 
        // For simplicity based on request, we rely on the model boot logic or auth user.
        // Actually, let's explicitly set it if not present, just in case model boot needs help or for clarity.
        if (empty($request->company_code)) {
            $validated['company_code'] = auth()->user()->company_code ?? session('company_code');
        }

        MetodePembayaran::create($validated);

        return redirect()->route('metodepembayaran.index')->with('success', 'Data berhasil ditambahkan');
    }

    public function edit($id)
    {
        $metodePembayaran = MetodePembayaran::findOrFail($id);

        $user = auth()->user();
        $tenant = $user->company_code ?? session('company_code');

        if ($tenant !== '000' && $metodePembayaran->company_code !== $tenant) {
            abort(403, 'Unauthorized action.');
        }

        $breadcrumb = [
            ['name' => 'Home', 'url' => route('dashboard.index')],
            ['name' => 'Metode Pembayaran', 'url' => route('metodepembayaran.index')],
            ['name' => 'Edit', 'url' => ''],
        ];

        return view('metodepembayaran.edit', compact('metodePembayaran', 'breadcrumb'));
    }

    public function update(Request $request, $id)
    {
        $metodePembayaran = MetodePembayaran::findOrFail($id);

        $user = auth()->user();
        $tenant = $user->company_code ?? session('company_code');

        if ($tenant !== '000' && $metodePembayaran->company_code !== $tenant) {
            abort(403, 'Unauthorized action.');
        }

        $validated = $request->validate([
            'metode_pembayaran' => 'required|string|max:100',
            'keterangan' => 'nullable|string',
        ]);

        $metodePembayaran->update($validated);

        return redirect()->route('metodepembayaran.index')->with('success', 'Data berhasil diperbarui');
    }

    public function destroy($id)
    {
        $metodePembayaran = MetodePembayaran::findOrFail($id);

        $user = auth()->user();
        $tenant = $user->company_code ?? session('company_code');

        if ($tenant !== '000' && $metodePembayaran->company_code !== $tenant) {
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json(['success' => false, 'message' => 'Unauthorized'], 403);
            }
            abort(403);
        }

        $metodePembayaran->delete();

        if (request()->ajax() || request()->wantsJson()) {
            return response()->json(['success' => true, 'message' => 'Data berhasil dihapus']);
        }

        return redirect()->route('metodepembayaran.index')->with('success', 'Data berhasil dihapus');
    }
}
