<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Pelanggan;
use Illuminate\Support\Facades\Auth;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Illuminate\Support\Facades\DB;

class ImportDataPelangganController extends Controller
{
    public function index()
    {
        $breadcrumb = [
            ['name' => 'Home', 'url' => route('dashboard.index')],
            ['name' => 'Pelanggan', 'url' => route('pelanggan.index')],
            ['name' => 'Import', 'url' => ''],
        ];

        return view('importdatapelanggan.index', compact('breadcrumb'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xlsx,xls,csv|max:2048',
        ]);

        try {
            $file = $request->file('file');
            $spreadsheet = IOFactory::load($file->getPathname());
            $worksheet = $spreadsheet->getActiveSheet();
            $rows = $worksheet->toArray();

            // Assuming first row is header
            $header = array_shift($rows);

            // Map headers to column indices if possible, or just use fixed indices. 
            // For now, let's assume specific columns or just map strictly by index if headers match what created. 
            // Better approach: Let's assume the excel should have headers matching DB columns for simplicity, 
            // or we define a fixed expected structure.
            // Let's implement a loop that tries to map common fields.

            DB::beginTransaction();

            $count = 0;
            foreach ($rows as $row) {
                // Skip empty rows
                if (empty($row[0])) continue;

                // Basic mapping based on column order
                // 0: Nama, 1: Alamat, 2: HP, 3: Paket Product ID, 4: Username PPPPoe, 5: Password
                // Adjust this mapping as per real requirement later.

                // Map Status
                $rawStatus = $row[10] ?? 'active';
                $status = 'active'; // Default
                switch (strtolower($rawStatus)) {
                    case 'aktif':
                    case 'active':
                        $status = 'active';
                        break;
                    case 'nonaktif':
                    case 'inactive':
                    case 'suspend':
                        $status = 'inactive';
                        break;
                    default:
                        $status = 'inactive';
                        break;
                }

                Pelanggan::create([
                    'nama' => $row[0] ?? null,
                    'alamat' => $row[1] ?? null,
                    'hp' => $row[2] ?? null,
                    'paket_product' => $row[3] ?? null, // Expecting ID
                    'username_pppoe' => $row[4] ?? null,
                    'password' => $row[5] ?? null,
                    'tgl_pendaftaran' => $row[6] ?? date('Y-m-d'),
                    'tgl_pasang' => $row[7] ?? null,
                    'tagihan_dimulai_tgl' => $row[8] ?? null,
                    'map' => $row[9] ?? null,
                    'status' => $status,
                    'keterangan' => $row[11] ?? null,
                    'company_code' => Auth::user()->company_code ?? '000',
                ]);
                $count++;
            }

            DB::commit();

            return redirect()->route('pelanggan.index')->with('success', "Berhasil mengimport $count data pelanggan.");
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Terjadi kesalahan saat import: ' . $e->getMessage());
        }
    }

    public function downloadTemplate()
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Set Headers
        $headers = [
            'Nama',
            'Alamat',
            'HP',
            'ID Paket Product',
            'Username PPPoE',
            'Password',
            'Tanggal Pendaftaran (YYYY-MM-DD)',
            'Tanggal Pasang (YYYY-MM-DD)',
            'Tanggal Tagihan Dimulai (YYYY-MM-DD)',
            'Map (Koordinat)',
            'Status',
            'Keterangan'
        ];

        $column = 'A';
        foreach ($headers as $header) {
            $sheet->setCellValue($column . '1', $header);
            $sheet->getColumnDimension($column)->setAutoSize(true);
            $column++;
        }

        // Add 5 sample data examples
        $samples = [
            [
                'Budi Santoso',
                'Jl. Merdeka No. 10',
                '081234567890',
                '1',
                'budi.santoso',
                'rahasia123',
                date('Y-m-d'),
                date('Y-m-d'),
                date('Y-m-d'),
                '-7.6298, 111.5239',
                'Aktif',
                'Data Pelanggan Baru'
            ],
            [
                'Siti Aminah',
                'Jl. Kenanga No. 5',
                '081398765432',
                '2',
                'siti.aminah',
                'aman456',
                date('Y-m-d', strtotime('-1 day')),
                date('Y-m-d', strtotime('-1 day')),
                date('Y-m-d', strtotime('-1 day')),
                '-7.6350, 111.5300',
                'Aktif',
                'Pindahan dari provider lain'
            ],
            [
                'Ahmad Rizki',
                'Jl. Mawar No. 88',
                '081567890123',
                '1',
                'ahmad.rizki',
                'pass789',
                date('Y-m-d', strtotime('-2 days')),
                date('Y-m-d', strtotime('-2 days')),
                date('Y-m-d', strtotime('-2 days')),
                '-7.6250, 111.5200',
                'Nonaktif',
                'Menunggu perangkat'
            ],
            [
                'Dewi Lestari',
                'Jl. Melati No. 22',
                '081987654321',
                '3',
                'dewi.lestari',
                'word321',
                date('Y-m-d', strtotime('-5 days')),
                date('Y-m-d', strtotime('-5 days')),
                date('Y-m-d', strtotime('-5 days')),
                '-7.6400, 111.5150',
                'Aktif',
                'Upgrade paket'
            ],
            [
                'Eko Prasetyo',
                'Jl. Anggrek No. 15',
                '085234567890',
                '2',
                'eko.prasetyo',
                'secure555',
                date('Y-m-d', strtotime('-10 days')),
                date('Y-m-d', strtotime('-10 days')),
                date('Y-m-d', strtotime('-10 days')),
                '-7.6200, 111.5350',
                'Aktif',
                'Telat bayar'
            ],
        ];

        $row = 2;
        foreach ($samples as $sample) {
            $col = 'A';
            foreach ($sample as $value) {
                $sheet->setCellValue($col . $row, $value);
                $col++;
            }
            $row++;
        }

        $writer = new Xlsx($spreadsheet);

        $fileName = 'template_import_pelanggan.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment; filename="' . $fileName . '"');
        header('Cache-Control: max-age=0');

        $writer->save('php://output');
        exit;
    }
}
