<?php

namespace App\Http\Controllers;

use App\Models\Company;
use Illuminate\Http\Request;

class CompanyController extends Controller
{
    public function index(Request $request)
    {
        $user = auth()->user();
        $tenant = $user?->company_code ?? session('company_code');
        $userCompanyCode = $tenant;
        $role = $user?->role;

        // Check read permission (Superadmin 000 bypasses check)
        if ($tenant !== '000') {
            if (!$role || !$role->hak_akses_tbl_company_read) {
                abort(403, 'Anda tidak memiliki akses untuk melihat data company.');
            }
        }

        $limit = $request->input('limit', 40);
        $search = $request->input('search');

        $companies = Company::query()
            ->when($tenant && $tenant !== '000', function ($q) use ($tenant) {
                $q->where('company_code', $tenant);
            })
            ->when($search, function ($query, $search) {
                return $query->where('company_code', 'like', "%{$search}%")
                    ->orWhere('nama_company', 'like', "%{$search}%")
                    ->orWhere('status', 'like', "%{$search}%")
                    ->orWhere('alamat', 'like', "%{$search}%")
                    ->orWhere('hp', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%");
            })
            ->paginate($limit)
            ->withQueryString();

        // Permission Checks based on Role & Company Constraints
        // Superadmin 000 has full access
        if ($tenant === '000') {
            $canCreate = true;
            $canEdit = true;
            $canDelete = true;
        } else {
            // Regular users checked against role
            // Create: Role bit = 1 AND Company = 000 (Regular users usually can't create companies unless logic changes)
            // But per previous strict rules: create/delete usually restricted to 000 context.
            // If we follow strictly "if not 000, follow access rights":
            $canCreate = ($role && $role->hak_akses_tbl_company_cread);
            $canEdit = ($role && $role->hak_akses_tbl_company_edit);
            $canDelete = ($role && $role->hak_akses_tbl_company_delete);
        }

        $breadcrumb = [
            ['name' => 'Home', 'url' => route('dashboard.index')],
            ['name' => 'Company', 'url' => ''],
        ];

        return view('company.index', compact('companies', 'breadcrumb', 'userCompanyCode', 'canCreate', 'canEdit', 'canDelete'));
    }

    public function create()
    {
        $user = auth()->user();
        $tenant = $user->company_code ?? session('company_code');
        $role = $user->role;

        // Check create permission: Superadmin 000 bypasses check
        if ($tenant !== '000') {
            if (!$role || !$role->hak_akses_tbl_company_cread) {
                return redirect()->route('company.index')->with('error', 'Anda tidak memiliki akses untuk menambah company.');
            }
        }

        $breadcrumb = [
            ['name' => 'Home', 'url' => route('dashboard.index')],
            ['name' => 'Company', 'url' => route('company.index')],
            ['name' => 'Tambah', 'url' => ''],
        ];

        return view('company.create', compact('breadcrumb'));
    }

    public function store(Request $request)
    {
        $user = auth()->user();
        $tenant = $user->company_code ?? session('company_code');
        $role = $user->role;

        // Check create permission: Superadmin 000 bypasses check
        if ($tenant !== '000') {
            if (!$role || !$role->hak_akses_tbl_company_cread) {
                return redirect()->route('company.index')->with('error', 'Anda tidak memiliki akses untuk menambah company.');
            }
        }

        $validated = $request->validate([
            'company_code' => 'required|unique:tbl_company',
            'nama_company' => 'required',
            'status' => 'required',
            'expired_at' => 'nullable|date',
            'alamat' => 'nullable',
            'hp' => 'nullable',
            'email' => 'nullable|email',
            'logo' => 'nullable|file|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        if ($request->hasFile('logo')) {
            $validated['logo'] = $request->file('logo')->store('logos', 'public');
        }

        Company::create($validated);

        return redirect()->route('company.index')->with('success', 'Data berhasil ditambahkan');
    }

    public function edit($id)
    {
        $user = auth()->user();
        $tenant = $user->company_code ?? session('company_code');
        $role = $user?->role;

        // Check edit permission: Superadmin 000 bypasses check
        if ($tenant !== '000') {
            if (!$role || !$role->hak_akses_tbl_company_edit) {
                return redirect()->route('company.index')->with('error', 'Anda tidak memiliki akses untuk mengedit company.');
            }
        }

        $company = Company::findOrFail($id);
        if ($tenant && $tenant !== '000' && $company->company_code !== $tenant) {
            return redirect()->route('company.index')->with('error', 'Akses ditolak');
        }
        $breadcrumb = [
            ['name' => 'Home', 'url' => route('dashboard.index')],
            ['name' => 'Company', 'url' => route('company.index')],
            ['name' => 'Edit', 'url' => ''],
        ];

        return view('company.edit', compact('company', 'breadcrumb'));
    }

    public function update(Request $request, $id)
    {
        $user = auth()->user();
        $tenant = $user->company_code ?? session('company_code');
        $role = $user?->role;

        // Check update permission: Superadmin 000 bypasses check
        if ($tenant !== '000') {
            if (!$role || !$role->hak_akses_tbl_company_edit) {
                return redirect()->route('company.index')->with('error', 'Anda tidak memiliki akses untuk mengedit company.');
            }
        }

        $company = Company::findOrFail($id);
        if ($tenant && $tenant !== '000' && $company->company_code !== $tenant) {
            return redirect()->route('company.index')->with('error', 'Akses ditolak');
        }

        $validated = $request->validate([
            'company_code' => 'required|unique:tbl_company,company_code,' . $id,
            'nama_company' => 'required',
            'status' => 'required',
            'expired_at' => 'nullable|date',
            'alamat' => 'nullable',
            'hp' => 'nullable',
            'email' => 'nullable|email',
            'logo' => 'nullable|file|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        if ($request->hasFile('logo')) {
            $validated['logo'] = $request->file('logo')->store('logos', 'public');
        }

        $company->update($validated);

        return redirect()->route('company.index')->with('success', 'Data berhasil diperbarui');
    }

    public function destroy($id)
    {
        $user = auth()->user();
        $tenant = $user?->company_code ?? session('company_code');
        $role = $user?->role;

        // Check delete permission: Superadmin 000 bypasses check
        if ($tenant !== '000') {
            if (!$role || !$role->hak_akses_tbl_company_delete) {
                if (request()->ajax() || request()->wantsJson()) {
                    return response()->json(['success' => false, 'message' => 'Anda tidak memiliki akses untuk menghapus company.'], 403);
                }
                return redirect()->route('company.index')->with('error', 'Anda tidak memiliki akses untuk menghapus company.');
            }
        }

        $company = Company::findOrFail($id);

        // Prevent deletion of super admin company
        if ($company->company_code === '000') {
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json(['success' => false, 'message' => 'Company dengan kode 000 tidak boleh dihapus'], 403);
            }
            return redirect()->route('company.index')->with('error', 'Company dengan kode 000 tidak boleh dihapus');
        }

        $company->delete();

        if (request()->ajax() || request()->wantsJson()) {
            return response()->json(['success' => true, 'message' => 'Data berhasil dihapus']);
        }

        return redirect()->route('company.index')->with('success', 'Data berhasil dihapus');
    }
}
