<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use App\Models\Company;

class BackupDatabaseController extends Controller
{
    public function index()
    {
        $user = auth()->user();
        $tenant = $user->company_code ?? '000';
        $isSuperAdmin = ($tenant === '000');

        $breadcrumb = [
            ['name' => 'Dashboard', 'url' => route('dashboard.index')],
            ['name' => 'Backup Database', 'url' => route('backupdatabase.index')],
        ];

        // Ensure storage folder exists
        if (!file_exists(storage_path('app/backups'))) {
            mkdir(storage_path('app/backups'), 0777, true);
        }

        $files = [];
        foreach (glob(storage_path('app/backups/*.sql')) as $path) {
            $fileName = basename($path);

            // Filter: If not superadmin, only show own prefix
            if (!$isSuperAdmin) {
                // Check prefix "backup_TENANT-"
                if (strpos($fileName, 'backup_' . $tenant . '-') !== 0) {
                    continue;
                }
            }

            $files[] = [
                'name' => $fileName,
                'path' => $path,
                'size' => round(filesize($path) / 1024 / 1024, 2) . ' MB',
                'date' => date('Y-m-d H:i:s', filemtime($path))
            ];
        }

        // Sort by date desc
        usort($files, function ($a, $b) {
            return strtotime($b['date']) - strtotime($a['date']);
        });

        return view('backupdatabase.index', compact('breadcrumb', 'files'));
    }

    public function backup()
    {
        $user = auth()->user();
        $tenant = $user->company_code ?? '000';
        $isSuperAdmin = ($tenant === '000');

        // Determine filename with tenant info
        $companyName = 'Superadmin';
        if (!$isSuperAdmin) {
            $company = Company::where('company_code', $tenant)->first();
            if ($company) {
                // Sanitize company name for filename (replace non-alphanumeric with _)
                $cleanName = preg_replace('/[^A-Za-z0-9\-]/', '_', $company->nama_company);
                $companyName = $cleanName;
            } else {
                $companyName = 'Unknown';
            }
        }

        $fileName = 'backup_' . $tenant . '-' . $companyName . '_' . date('Y-m-d_H-i-s') . '.sql';
        $path = storage_path('app/backups/' . $fileName);

        // Ensure storage folder exists
        if (!file_exists(dirname($path))) {
            mkdir(dirname($path), 0777, true);
        }

        $tables = DB::select('SHOW TABLES');
        $dbName = DB::connection()->getDatabaseName();

        $content = "-- Database Backup\n";
        $content .= "-- Filename: " . $fileName . "\n";
        $content .= "-- Generated: " . date('Y-m-d H:i:s') . "\n";
        $content .= "-- User: " . ($user->name ?? 'Unknown') . " (Tenant: $tenant)\n\n";
        $content .= "SET FOREIGN_KEY_CHECKS=0;\n\n";

        foreach ($tables as $tableObj) {
            $table = $tableObj->{'Tables_in_' . $dbName};

            // Get columns
            $columns = Schema::getColumnListing($table);
            $hasCompanyCode = in_array('company_code', $columns);

            // If tenant, only dump tables with company_code
            if (!$isSuperAdmin && !$hasCompanyCode) {
                continue;
            }

            // Structure
            // Only Superadmin gets structural dumps (DROP/CREATE)
            if ($isSuperAdmin) {
                $content .= "-- Table structure for `$table`\n";
                $content .= "DROP TABLE IF EXISTS `$table`;\n";
                $createTable = DB::select("SHOW CREATE TABLE `$table`")[0]->{'Create Table'};
                $content .= $createTable . ";\n\n";
            }

            // Data
            $content .= "-- Dumping data for table `$table`\n";

            $query = DB::table($table);
            if (!$isSuperAdmin && $hasCompanyCode) {
                $query->where('company_code', $tenant);

                // Add delete statement for clean import of tenant data
                $content .= "DELETE FROM `$table` WHERE `company_code` = '$tenant';\n";
            }

            foreach ($query->cursor() as $row) {
                $row = (array) $row;
                $values = array_map(function ($value) {
                    return is_null($value) ? 'NULL' : DB::getPdo()->quote($value);
                }, $row);

                $content .= "INSERT INTO `$table` VALUES (" . implode(", ", $values) . ");\n";
            }
            $content .= "\n\n";
        }

        $content .= "SET FOREIGN_KEY_CHECKS=1;\n";

        file_put_contents($path, $content);

        return redirect()->back()->with('success', 'Backup berhasil dibuat dan disimpan di server: ' . $fileName);
    }

    public function download($fileName)
    {
        $user = auth()->user();
        $tenant = $user->company_code ?? '000';

        // Security check for tenants
        if ($tenant !== '000') {
            if (strpos($fileName, 'backup_' . $tenant . '-') !== 0) {
                return redirect()->back()->with('error', 'Anda tidak memiliki akses ke file backup ini.');
            }
        }

        $path = storage_path('app/backups/' . $fileName);
        if (!file_exists($path)) {
            return redirect()->back()->with('error', 'File tidak ditemukan');
        }
        return response()->download($path);
    }

    public function destroy($fileName)
    {
        $user = auth()->user();
        $tenant = $user->company_code ?? '000';

        // Security check for tenants
        if ($tenant !== '000') {
            if (strpos($fileName, 'backup_' . $tenant . '-') !== 0) {
                return redirect()->back()->with('error', 'Anda tidak memiliki akses untuk menghapus file ini.');
            }
        }

        $path = storage_path('app/backups/' . $fileName);
        if (file_exists($path)) {
            unlink($path);
            return redirect()->back()->with('success', 'File backup berhasil dihapus');
        }
        return redirect()->back()->with('error', 'File tidak ditemukan');
    }

    public function restore(Request $request)
    {
        $user = auth()->user();
        $tenant = $user->company_code ?? '000';
        $isSuperAdmin = ($tenant === '000');

        $request->validate([
            'sql_file' => 'required|file|mimes:sql,txt',
        ]);

        $file = $request->file('sql_file');

        ini_set('memory_limit', '-1');
        set_time_limit(0);

        try {
            $sql = file_get_contents($file->getRealPath());

            // Check for dangerous DDL commands if not Superadmin
            if (!$isSuperAdmin) {
                // List of DDL keywords to block for tenants
                $dangerousKeywords = ['DROP TABLE', 'CREATE TABLE', 'ALTER TABLE', 'TRUNCATE', 'DROP DATABASE'];

                foreach ($dangerousKeywords as $keyword) {
                    // Use case-insensitive check
                    if (stripos($sql, $keyword) !== false) {
                        return redirect()->back()->with('error', "Keamanan: Tenant tidak diizinkan menjalankan perintah '$keyword'. Gunakan file backup khusus tenant.");
                    }
                }
            }

            DB::unprepared($sql);

            return redirect()->back()->with('success', 'Database berhasil diimport!');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Gagal mengimport database: ' . $e->getMessage());
        }
    }
}
